/*
   Gracias al autor original: ryand1011 (https://github.com/ryand1011)
   Edited by francio7s1 -

   Ver: https://github.com/miguelbalboa/rfid/tree/master/examples/rfid_write_personal_data

   Usa tarjetas MIFARE RFID usando un lector RFID-RC522
   Usa la librería MFRC522
   -----------------------------------------------------------------------------------------
               MFRC522      Arduino       Arduino   Arduino    Arduino          Arduino
               Reader/PCD   Uno/101       Mega      Nano v3    Leonardo/Micro   Pro Micro
   Signal      Pin          Pin           Pin       Pin        Pin              Pin
   -----------------------------------------------------------------------------------------
   RST/Reset   RST          9             5         D9         RESET/ICSP-5     RST
   SPI SS      SDA(SS)      10            53        D10        10               10
   SPI MOSI    MOSI         11 / ICSP-4   51        D11        ICSP-4           16
   SPI MISO    MISO         12 / ICSP-1   50        D12        ICSP-1           14
   SPI SCK     SCK          13 / ICSP-3   52        D13        ICSP-3           15
*/

#include <SPI.h>
#include <MFRC522.h>
#include "Arduino.h"
#include "SoftwareSerial.h"
#include "DFRobotDFPlayerMini.h"

// Pin de reset del MFRC522.
#define RST_PIN         9

// Pin de selección SPI del MFRC522.
#define SS_PIN          10

// Controles de volumen
const byte volumePot = A0;
int prevVolume; 
byte volumeLevel = 0;

boolean isPlaying = false;

// Objeto para controlar el lector RFID.
MFRC522 mfrc522(SS_PIN, RST_PIN);

// Comunicación serie por software (RX, TX) para DFPlayer.
SoftwareSerial mySoftwareSerial(5, 6);

// Objeto para controlar el DFPlayer.
DFRobotDFPlayerMini myDFPlayer;


void setup() {
  // Inicializa comunicación serie (PC).
  Serial.begin(115200);

  // Inicializa comunicación serie software (DFPlayer).
  mySoftwareSerial.begin(9600);

  // Inicializa bus SPI (para RFID).
  SPI.begin();

  // Inicializa lector RFID.
  mfrc522.PCD_Init();

  Serial.println(F("Iniciando DFPlayer ... (Puede llevar 3~5 segundos)"));

  // Inicializa el MP3, verifica conexión
  if (!myDFPlayer.begin(mySoftwareSerial)) { 
    Serial.println(F("No se ha podido iniciar:"));
    Serial.println(F("1.Por favor comprueb la conexión!"));
    Serial.println(F("2.Por favor inserte la tarjeta SD!"));
  }

  Serial.println(F("DFPlayer Mini activo. Coloque una tarjeta en el lector para reproducir un audio específico"));

  // Lectura inicial del volumen desde el potenciómetro, lo normaliza (0-30) y lo aplica al DFPlayer.
  volumeLevel = map(analogRead(volumePot), 0, 1023, 0, 30);
  myDFPlayer.volume(volumeLevel);
  prevVolume = volumeLevel;

  // Ecualizador del reproductor
  myDFPlayer.EQ(DFPLAYER_EQ_NORMAL);
}



//*****************************************************************************************//
void loop() {

  // Control del volumen
  volumeLevel = map(analogRead(volumePot), 0, 1023, 0, 30);
  
  if (prevVolume != volumeLevel){
    myDFPlayer.volume(volumeLevel);
  }

  prevVolume = volumeLevel;

  /*
    key es una clave que da acceso a los distintos sectores de una tarjeta RFID MIFARE.

    Las tarjetas MIFARE Classic están divididas en sectores y cada sector tiene dos claves: Key A y Key B (cada una de 6 bytes). 
    Estas claves controlan los permisos de lectura, escritura y acceso.

    La clave por defecto que traen las tarjetas nuevas es 0xFF 0xFF 0xFF 0xFF 0xFF 0xFF.

    Dentro del objeto key existe una variable llamada keyByte que contiene un array de 6 bytes con los valores de la clave.

    Para poder leer y escribir en las tarjetas necesitamos establecer esos 6 bytes a los valores de fábrica (ya que nadie ha cambiado los valores de las claves).

    Así que el bucle recorre el array de 6 bytes keyBytes y establece cada uno de los elementos del array al valor 0xFF
  */
  MFRC522::MIFARE_Key key;
  for (byte i = 0; i < 6; i++) key.keyByte[i] = 0xFF;

  // Número del bloque de la tarjeta RFID que se va a leer.
  byte block;

  // Indica cuántos bytes se esperan leer y también servirá para que la función MIFARE_Read() devuelva cuántos bytes realmente se leyeron.
  byte len;

  // Estado que indica si la autenticación con el bloque de la tarjeta RFID fue exitosa o falló.
  MFRC522::StatusCode status;

  //-------------------------------------------

  // Comprueba si se ha colocado una nueva tarjeta.
  if ( mfrc522.PICC_IsNewCardPresent()) {

    // Lee la tarjeta, y en caso de fallo de lectura sale de la función.
    if ( !mfrc522.PICC_ReadCardSerial()) {
      return;
    }

    Serial.println(F("**Tarjeta detectada:**"));

    // Muestra los detalles de la tarjeta por el monitor serie.
    mfrc522.PICC_DumpDetailsToSerial(&(mfrc522.uid));

    Serial.print(F("Número: "));

    //  Array de 18 bytes que servirá como memoria temporal para guardar los datos leídos de un bloque de la tarjeta RFID.
    byte buffer2[18];

    // Establece que el bloque se va a leer de la tarjeta es el 1
    block = 1;

    // Establece que se esperan leer 18 bytes
    len = 18;

    /*
      Para poder leer y escribir de la tarjeta necesitamos un MFRC522::STATUS_OK y para ello tenemos que primero "autenticarnos".

      Así que usaremos la función PCD_Authenticate que recibe los siguientes parámetros:
        - MFRC522::PICC_CMD_MF_AUTH_KEY_A -> indica qué clave, si la A o la B, vamos a usar para acceder al sector que corresponda.
        - 1 -> Bloque de la tarjeta al que se intenta acceder, en este caso al 1.
        - key -> clave de 6 bytes (que hemos establecido antes) que se utilizarán para autenticarse. Al estar con & se accede directamente
          a la dirección de memoria dónde se alamacena dicha clave.
        - mfrc522.uid -> Identificador único de la tarjeta. Al estar con & se accede directamente a la dirección de memoria dónde se 
        alamacena dicho identificador.

      Si la identificación es exitosa, se puede leer o escribir en el bloque deseado
    */
    status = mfrc522.PCD_Authenticate(MFRC522::PICC_CMD_MF_AUTH_KEY_A, 1, &key, &(mfrc522.uid));

    // Comprueba que la autenticación haya sido correcta, y de no ser así informa de que ha habido un error y del status correspondiente.
    if (status != MFRC522::STATUS_OK) {
      Serial.print(F("Fallo de autenticación: "));
      Serial.println(mfrc522.GetStatusCodeName(status));
      return;
    }

    // Lee los datos de la tarjeta, indicando el bloque que tiene que leer, el lugar donde alamacenará lo leido, y como de grande se espera que sea
    status = mfrc522.MIFARE_Read(block, buffer2, &len);

    // Comprueba que la lectura haya sido correcta, y de no ser así informa de que ha habido un error y del status correspondiente.
    if (status != MFRC522::STATUS_OK) {
      Serial.print(F("Fallo de lectura: "));
      Serial.println(mfrc522.GetStatusCodeName(status));
      return;
    }


    // Variable que alamacenará el número que contiene la tarjeta, el que corresponde con nuestro audio
    String number = "";

    /*
      Este bucle recorre los 16 bytes del array buffer2.

      El array buffer2 es la variable donde esta almacenada la información que hemos leido de la tarjeta.

      Cada uno de los bytes (en hexadecimal) que va leyendo los convierte a caracteres con la función (char).

      Una vez convertido los concatena dentro de la variable number (number no es un número, es una cadena de texto)

      Cuando termina de recorrer el bucle se usa la función trim para quitar espacios en blanco.

      De esta manera en la variable number se almacena el número correspondiente a nuestra tarjeta.
    */
    for (uint8_t i = 0; i < 16; i++) {
      number += (char)buffer2[i];
    }
    number.trim();
    Serial.print(number);

    // Reproduce la canción correspondiente al número que hemos obtenido (convierte la cadena de texto a un número entero).
    myDFPlayer.play(number.toInt());
    isPlaying = true;



    //----------------------------------------

    Serial.println(F("\n**Fin de lectura**\n"));

    delay(1000);

    mfrc522.PICC_HaltA();
    mfrc522.PCD_StopCrypto1();
  }
}
