/*
   Escribe el número de audio a una tarjeta MIFARE RFID usando un lector RFID-RC522
   Usa la libreria MFRC52 para usar el ARDUINO RFID MODULE KIT 13.56 MHZ WITH TAGS SPI W AND R BY COOQROBOT.
   -----------------------------------------------------------------------------------------
               MFRC522      Arduino       Arduino   Arduino    Arduino          Arduino
               Reader/PCD   Uno/101       Mega      Nano v3    Leonardo/Micro   Pro Micro
   Signal      Pin          Pin           Pin       Pin        Pin              Pin
   -----------------------------------------------------------------------------------------
   RST/Reset   RST          9             5         D9         RESET/ICSP-5     RST
   SPI SS      SDA(SS)      10            53        D10        10               10
   SPI MOSI    MOSI         11 / ICSP-4   51        D11        ICSP-4           16
   SPI MISO    MISO         12 / ICSP-1   50        D12        ICSP-1           14
   SPI SCK     SCK          13 / ICSP-3   52        D13        ICSP-3           15

   Hardware necesario:
   Arduino
   PCD
   PICC
*/

#include <SPI.h>
#include <MFRC522.h>

// Pin conectado al Reset del lector RC522 (permite reiniciarlo por software).
#define RST_PIN         9

// Pin Slave Select para la comunicación SPI, selecciona el dispositivo al que enviar datos.
#define SS_PIN          10

// Variable donde se alamcenará si el modo que ha elegido el usuario es automático o manual
String mode;

// En caso de elegir modo automático, este será el número que se escribirá en la tarjeta. Se incrementará postriormente.
int currentNumber = 1;

// Crea el objeto para controlar el lector RC522 indicando qué pines usa para SS y RST.
MFRC522 mfrc522(SS_PIN, RST_PIN);

void setup() {
  // Iniciar comunicación serie a 115200 baudios
  Serial.begin(115200);

  // Iniciar el bus SPI
  SPI.begin();

  // Inicializar el lector RFID
  mfrc522.PCD_Init();

  Serial.println(F("Escribe manual para el modo manual, o auto para el modo automático"));
}


void loop() {

  // Comprueba si hay algo escrito en el puerto serie
  if (Serial.available()) {

    // Almacena en la variable mode el modo que ha elegido el usuario. Lee lo que hay escrito en el puerto serie hasta el primer salto de linea
    mode = Serial.readStringUntil('\n');

    // Comprueba que lo que se haya escrito es auto o manual
    if (mode == "auto" || "manual") {
      Serial.println(" ");
      Serial.println("Ahora el dispositivo se encuentra en modo " + mode);
      Serial.println(F("Coloca la tarjeta sobre el lector para poder grabar en ella el número de audio."));

      // Solo si el modo es automático
      if (mode == "auto") Serial.println((String)"Número que se va a escribir en la tarjeta: " + currentNumber);

    }

    // Si el modo es automático se llama a la función que gestiona el modo automático
    while (mode == "auto") {
      autoModus();
    }

    // Si el modo es manual se llama a la función que gestiona el modo manual
    while (mode == "manual") {
      manualModus();
    }

    // Si lo que se ha escrito no es ni manual ni auto se le informa al usuario y se pide que vuelva a escribir un valor otra vez
    if (mode != "auto" || "manual") {
      Serial.println("Valor incorrecto, escriba auto o manual");
    }

  }
}

void autoModus() {



  // Establece la clave a 0xFF 0xFF 0xFF 0xFF 0xFF 0xFF. Ver explicación en Reproductor-parlanchinator
  MFRC522::MIFARE_Key key;
  for (byte i = 0; i < 6; i++) key.keyByte[i] = 0xFF;

  // Detecta si se ha colocado una nueva tarjeta en el lector
  if ( mfrc522.PICC_IsNewCardPresent()) {

    Serial.println(" ");

    // Si no se puede leer la tarjeta sale de la función
    if ( ! mfrc522.PICC_ReadCardSerial()) {
      return;
    }

    // Muestra el identificador único de la tarjeta
    Serial.print(F("UID de la tarjeta:"));
    /*
      El identificador de UID de la tarjeta se encuentra en un array de bytes.

      Se recorre dicho array, y se muestran los bytes en hexadecimal
    */
    for (byte i = 0; i < mfrc522.uid.size; i++) {
      Serial.print(mfrc522.uid.uidByte[i] < 0x10 ? " 0" : " ");
      Serial.print(mfrc522.uid.uidByte[i], HEX);
    }

    /*
      PICC significa Proximity Integrated Circuit Card (Tarjeta de Circuito Integrado de Proximidad).
      Es el término técnico para referirse a la tarjeta o etiqueta RFID que se acerca al lector para comunicación sin contacto.
      En el contexto MIFARE y RFID, el PICC es la tarjeta física, que cumple con el estándar ISO14443A para tarjetas de proximidad (13.56 MHz).
      Cuando acercas la tarjeta al lector RFID, el lector detecta un PICC.

      SAK significa Select Acknowledge (Reconocimiento de selección). Es un byte que la tarjeta PICC envía al lector durante el proceso de conexión.
      El SAK contiene información sobre el tipo y las capacidades de la tarjeta. 
      El lector usa el valor del SAK para identificar exactamente qué tipo de tarjeta está leyendo (por ejemplo, MIFARE 1K, MIFARE Ultralight, NTAG, etc.).

      mfrc522.uid.sak es ese valor extraído del UID de la tarjeta detectada.
    */
    Serial.print(F(" Tipo de PICC: "));

    // Obtiene el tipo de PICC de la tarjeta
    MFRC522::PICC_Type piccType = mfrc522.PICC_GetType(mfrc522.uid.sak);

    // Transforma el tipo de PICC de la tarjeta a una cadena de texto legible
    Serial.println(mfrc522.PICC_GetTypeName(piccType));

    // Varibale que va a almacenar el valor que se escribirá en la tarjeta  
    byte buffer[16];

    // Estado de las disitintas operaciones que se hagan con la tarjeta
    MFRC522::StatusCode status;

    // Bloque de la tarjeta en el que se escribirá
    byte block;

    // Longitud del mensaje que se escribirá en la tarjeta
    byte len;

    // Se establece que el bloque en el que se escribirá es en el 1
    block = 1;

    /*
      sprintf es una función que convierte datos en texto (cadena de caracteres) y los escribe en un array de caracteres (buffer).

      Sintaxis: sprintf(destino, "formato", valores).

      En esete caso el destino es buffer (dónde queremos escribir el valor), el formato es %d (entero en base decimal, sin decimales), y currentNumber es lo que queremos convertir.

      Además la función sprintf nos devuelve la longitud de la cadena escrita.

      Después se comienza un bucle a partir del primer byte que no está escrito (len)  y se rellenan con espacios vacíos hasta llegar a 30 bytes.

      De esta manera te aseguras que cada cadena tenga siempre la misma longitud
    */
    len = sprintf(buffer, "%d", currentNumber);
    for (byte i = len; i < 17; i++) buffer[i] = ' ';


    /*
      Para poder leer y escribir de la tarjeta necesitamos un MFRC522::STATUS_OK y para ello tenemos que primero "autenticarnos".

      Así que usaremos la función PCD_Authenticate que recibe los siguientes parámetros:
        - MFRC522::PICC_CMD_MF_AUTH_KEY_A -> indica qué clave, si la A o la B, vamos a usar para acceder al sector que corresponda.
        - block -> Bloque de la tarjeta al que se intenta acceder, en este caso al 1.
        - key -> clave de 6 bytes (que hemos establecido antes) que se utilizarán para autenticarse. Al estar con & se accede directamente
          a la dirección de memoria dónde se alamacena dicha clave.
        - mfrc522.uid -> Identificador único de la tarjeta. Al estar con & se accede directamente a la dirección de memoria dónde se 
        alamacena dicho identificador.

      Si la identificación es exitosa, se puede leer o escribir en el bloque deseado
    */
    status = mfrc522.PCD_Authenticate(MFRC522::PICC_CMD_MF_AUTH_KEY_A, block, &key, &(mfrc522.uid));

    // Si falla la autenticación sale de la función y muestra el status.
    if (status != MFRC522::STATUS_OK) {
      Serial.print(F("PCD_Authenticate() falló: "));
      Serial.println(mfrc522.GetStatusCodeName(status));
      return;
    }


    Serial.println(F("PCD_Authenticate() exitoso: "));

    // Escribe el en bloque correspondiente el buufer indicando que tendrá una longitud de 16 bytes
    status = mfrc522.MIFARE_Write(block, buffer, 16);

    // Sale de la función si hay algún fallo en la escritura
    if (status != MFRC522::STATUS_OK) {
      Serial.print(F("MIFARE_Write() falló: "));
      Serial.println(mfrc522.GetStatusCodeName(status));
      Serial.print(F("Prueba a reiniciar el dispositivo"));
      return;
    }
    
    
    Serial.println(F("MIFARE_Write() exitoso: "));


    mfrc522.PICC_HaltA();
    mfrc522.PCD_StopCrypto1();

    // Aumenta el número de pista para la siguiente iteración
    currentNumber++;
    Serial.println((String)"Ponga una nueva tarjeta en el lector para escribir el número: " + currentNumber);

  }
}


void manualModus() {

  MFRC522::MIFARE_Key key;
  for (byte i = 0; i < 6; i++) key.keyByte[i] = 0xFF;

  if ( ! mfrc522.PICC_IsNewCardPresent()) {
    return;
  }

  if ( ! mfrc522.PICC_ReadCardSerial()) {
    return;
  }

  Serial.print(F("UID de la tarjeta:"));
  for (byte i = 0; i < mfrc522.uid.size; i++) {
    Serial.print(mfrc522.uid.uidByte[i] < 0x10 ? " 0" : " ");
    Serial.print(mfrc522.uid.uidByte[i], HEX);
  }

  Serial.print(F(" Tipo de PICC: "));
  MFRC522::PICC_Type piccType = mfrc522.PICC_GetType(mfrc522.uid.sak);
  Serial.println(mfrc522.PICC_GetTypeName(piccType));

  byte buffer[34];
  byte block;
  MFRC522::StatusCode status;
  byte len;

  // Espera 20 segundos para que el usuario escriba un número
  Serial.setTimeout(20000L);

  // Pregunta por el número
  Serial.println(F("Escriba cualquier número y presione INTRO"));

  // Lee el número que ha escrito el usuario hasta 16 caractéres y lo almacena en la variable buffer. Además en len se obtiene la longitud de la cadena escrita
  len = Serial.readBytesUntil('\n', (char *) buffer, 30);

  // Se rellenan bytes restantes hasta llegar a 16 con espacios vacios
  for (byte i = len; i < 30; i++) buffer[i] = ' ';

  block = 1;

  status = mfrc522.PCD_Authenticate(MFRC522::PICC_CMD_MF_AUTH_KEY_A, block, &key, &(mfrc522.uid));

  if (status != MFRC522::STATUS_OK) {
    Serial.print(F("La autentificación falló: "));
    Serial.println(mfrc522.GetStatusCodeName(status));
    return;
  }
  else Serial.println(F("La autentificación fue correcta: "));

  status = mfrc522.MIFARE_Write(block, buffer, 16);
  if (status != MFRC522::STATUS_OK) {
    Serial.print(F("No se pudo grabar la tarjeta: "));
    Serial.println(mfrc522.GetStatusCodeName(status));
    return;
  }
  else Serial.println(F("Se grabó la tarjeta correctamente: "));

  Serial.println("Ponga una nueva tarjeta para escribir un nuevo número");



  Serial.println(" ");
  mfrc522.PICC_HaltA();
  mfrc522.PCD_StopCrypto1();

}
